/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <pthread.h>
#include <list>
#include <string>
#include <unistd.h>
#include <stdlib.h>
#include <iostream>
#include <fstream>
#include "Log.h"
#include "LogThread.h"

#define NO_LOG_THREADS 10
#define MESSAGE_CYCLES 100

using namespace std;

LOG_MODES parseLogMode(std::string& modeString)
{
    LOG_MODES fileLogLevel = LOG_INFO;

    if (!modeString.compare("LOG_DEBUG"))
    {
        fileLogLevel = LOG_DEBUG;
    }
    else if (!modeString.compare("LOG_ERROR"))
    {
        fileLogLevel = LOG_ERROR;
    }
    else if (!modeString.compare("LOG_INFO"))
    {
        fileLogLevel = LOG_INFO;
    }
    else if (!modeString.compare("LOG_WARNING"))
    {
        fileLogLevel = LOG_WARNING;
    }
    else if (!modeString.compare("LOG_DISABLED"))
    {
        fileLogLevel = LOG_DISABLED;
    }
    else
    {
        cerr << "Unknown Log Level: " << modeString << endl;
        cerr << "Should be: LOG_DEBUG, LOG_ERROR, LOG_INFO, "
                "LOG_WARNING or LOG_DISABLED" << endl;

        exit(0);
    }

    return fileLogLevel;
}

void usage(std::string& name)
{
    cerr << "Wrong Arguments for: " << name << endl;
    cerr << "Usage: " 
         << name
         << " -f<file log threshold>"
            " -c<console log threshold>"
            " -d<DLT log threshold>"
            " -t<number of threads>"
            " -C<number of log cycles>"
            " -F<results log name>"
         << endl;
}

struct testConfiguration
{
    unsigned int numThreads;
    unsigned int numMessageCycles;
    LOG_MODES fileLogLevel;
    LOG_MODES consoleLogLevel;
    LOG_MODES DLTLogLevel;
    std::string fileName;
    ofstream resultsFile;
};

bool parseCommandLine(int argc, const char* argv[], testConfiguration& config)
{
    std::string exeName(argv[0]);
    bool result = true;

    while ((argc > 1) && (argv[1][0] == '-'))
    {
        switch (argv[1][1])
        {
            case 'f':
                {
                    std::string fileLogThreshold(&argv[1][2]);
                    config.fileLogLevel = parseLogMode(fileLogThreshold);
                }
            break;

            case 'c':
                {
                    std::string consoleLogThreshold(&argv[1][2]);
                    config.consoleLogLevel = parseLogMode(consoleLogThreshold);
                }
            break;

            case 'd':
                {
                    std::string dltLogThreshold(&argv[1][2]);
                    config.DLTLogLevel = parseLogMode(dltLogThreshold);
                }
            break;

            case 't':
                {
                    std::string numThreads(&argv[1][2]);
                    istringstream iss(numThreads);
                    unsigned short int threads;
                    iss >> threads;
                    if (!iss.fail())
                    {
                        config.numThreads = threads;
                    }
                }
            break;

            case 'C':
                {
                    std::string numCycles(&argv[1][2]);
                    istringstream iss(numCycles);
                    unsigned short int cycles;
                    iss >> cycles;
                    if (!iss.fail())
                    {
                        config.numMessageCycles = cycles;
                    }
                }
            break;

            case 'F':
                {
                    std::string fileName(&argv[1][2]);
                    config.resultsFile.open(fileName.c_str());

                    if (!config.resultsFile.is_open())
                    {
                        cerr << "Unable to open file, reverting to standard output." << endl;
                    }
                }
            break;

            default:
                usage(exeName);
                result = false;
        }

        ++argv;
        --argc;
    }

    return result;

}

void initialiseDLT()
{
    system("systemctl start layer-management-wayland");
    system("systemctl stop layer-management-wayland");
}

int main( int argc, const char* argv[] )
{
    testConfiguration config;

    config.numThreads = NO_LOG_THREADS;
    config.numMessageCycles = MESSAGE_CYCLES;
    config.fileLogLevel = LOG_DEBUG;
    config.consoleLogLevel = LOG_DEBUG;
    config.DLTLogLevel = LOG_DEBUG;

    if (!parseCommandLine(argc,argv,config))
    {
        exit(0);
    }

    initialiseDLT();

    Log::getInstance()->setFileLogLevel(config.fileLogLevel);
    Log::getInstance()->setConsoleLogLevel(config.consoleLogLevel);
    Log::getInstance()->setDLTLogLevel(config.DLTLogLevel);

    if (config.resultsFile.is_open())
    {
        config.resultsFile << "Using: File Logging Level - "
                           << Log::getInstance()->logString[config.fileLogLevel]
                           << ", Console Logging Level - "
                           << Log::getInstance()->logString[config.consoleLogLevel]
                           << ", DLT Logging Level - "
                           << Log::getInstance()->logString[config.DLTLogLevel]
                            << endl;
    }
    else
    {
        cout << "Using: File Logging Level - " << Log::getInstance()->logString[config.fileLogLevel]
             << ", Console Logging Level - " << Log::getInstance()->logString[config.consoleLogLevel]
             << ", DLT Logging Level - " << Log::getInstance()->logString[config.DLTLogLevel] << endl;
    }

    std::list<LogThread*> listLogThreads;

    for (int i = 0; i < config.numThreads; i++)
    {
        LogThread* thread = new LogThread(i, config.numMessageCycles);
        listLogThreads.push_front(thread);
        pthread_create(&thread->getpThread(), NULL, LogThread::thread_start, thread);
    }

    for (std::list<LogThread*>::iterator itr = listLogThreads.begin();
         itr != listLogThreads.end();
         ++itr)
    {
        LogThread* pThreadInst = *itr;
        pthread_join(pThreadInst->getpThread(), NULL);
    }

    unsigned int cumulativeTestTime = 0;

    for (std::list<LogThread*>::iterator itr = listLogThreads.begin();
         itr != listLogThreads.end();
         ++itr)
    {
        LogThread* pThreadInst = *itr;
        cumulativeTestTime = cumulativeTestTime 
                             + pThreadInst->getCumulativeTime();
    }

    if (config.resultsFile.is_open())
    {
        config.resultsFile << "Test Results: "
                           << ", cumulative time: "
                           << cumulativeTestTime
                           << ", threads: "
                           << config.numThreads
                           << ", cycle done.: "
                           << config.numMessageCycles << endl;
    }
    else
    {
        cout << "Test Results: " <<
             ", cumulative time: " << cumulativeTestTime <<
             ", threads: " << config.numThreads <<
             ", cycle done.: " << config.numMessageCycles << endl;
    }

    if (config.resultsFile.is_open()) config.resultsFile.close();

    for (std::list<LogThread*>::iterator itr = listLogThreads.begin();
         itr != listLogThreads.end();
         ++itr)
    {
        LogThread* pThreadInst = *itr;
        delete pThreadInst;
    }

    Log::closeInstance();
    exit(0);
}
